function unmasked_sensor_data = unmaskSensorData(sensor_data, sensor_mask, kgrid)
%UNMASKSENSORDATA   Reorder data recorded using a binary sensor mask.
%
% DESCRIPTION:
%       unmaskSensorData restores the grid position of the data recorded at
%       a single time-step within the time-series data returned by
%       kspaceFirstOrder1D, kspaceFirstOrder2D, or kspaceFirstOrder3D when
%       using a binary sensor mask.
%
% USAGE:
%       unmasked_sensor_data = unmaskSensorData(sensor_data, sensor_mask, kgrid)
%
% INPUTS:
%       sensor_data - sensor data (returned by the first order simulation
%                     functions) at a single time-step ordered using
%                     MATLAB's standard column-wise linear matrix indexing 
%       sensor_mask - binary sensor mask used in the simulation
%       kgrid       - kspace grid structure returned by makeGrid
%
% OUTPUTS:
%       unmasked_sensor_data - Grid with the sensor data reordered to its
%                              original position on the sensor mask
%
% ABOUT:
%       author      - Bradley Treeby
%       date        - 6th April 2009
%       last update - 17th July 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
%
% See also makeGrid, kspaceFirstOrder1D, kspaceFirstOrder2D,
% kspaceFirstOrder3D

% create an empty matrix
switch numDim(kgrid.k)
    case 1
        unmasked_sensor_data = zeros(kgrid.Nx, 1);
    case 2
        unmasked_sensor_data = zeros(kgrid.Nz, kgrid.Nx);
    case 3
        unmasked_sensor_data = zeros(kgrid.Nz, kgrid.Nx, kgrid.Ny);
end

% reorder input data
unmasked_sensor_data(sensor_mask ~= 0) = sensor_data;